package service;

import java.util.Arrays;


public class ComposedStateImpl extends ComposedState {
	State[] states;
	boolean isInitial;
	boolean isFinal;
	/** Create a new ComposedState
	 * If all the state of the array states are initial states, the composed state will be a initial state
	 * If all the state of the array states are final states, the composed state will be a final state
	 * @param name name that identifies the state
	 * @param states array of states that make up the ComposedState
	 */
	public ComposedStateImpl(String name, State[] states) {
		/**
		 * ComposedState constructor assings only the name, in this constructor
		 * instead, we memorize what states contain
		 */
		super(name, states);
		this.states = states.clone();

	
		isInitial = true;
		isFinal = true;
		for (int i = 0; i < states.length; i++) {
			if (!states[i].isFinal())
				isFinal = false;
			if (!states[i].isInitial())
				isInitial = false;
		}
	}

	public ComposedStateImpl(State[] states) {
		super(states);
		this.states = states.clone();

		String name="";
		isInitial = true;
		isFinal = true;
		for (int i = 0; i < states.length; i++) {
			if (!states[i].isFinal())
				isFinal = false;
			if (!states[i].isInitial())
				isInitial = false;
			name=name+states[i].getName();
		}
		setName(name);
		
	}

	/** Returns the state of the ComposedState identified by the serviceNumber
	*/
	@Override
	public State getServicesState(int serviceNumber) throws Exception {
		if (serviceNumber >= states.length)
			throw new Exception(
					"Class: ComposedStateImpl Method: getServiceState Reason: illegalServiceNumber");
		return states[serviceNumber];
	}
	/** Returns the array of states that make up the ComposedState
	*/
	@Override
	public State[] getStates() {
		return states;
	}

	@Override
	/** Returns if the current state is final or not.
	 * @return returns true if the state is final else return false
	 */
	public boolean isFinal() {
		return isFinal;
	}
	/** Returns if the current state is a initial state.
	 * @return returns true if the state is a initials state else return false
	 */
	@Override
	public boolean isInitial() {
		return isInitial;
	}
	/**
	 * This method returns the hashCode of the state
	 * @return The string contains the hashCode of the state
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = super.hashCode();
		result = prime * result + (isFinal ? 1231 : 1237);
		result = prime * result + (isInitial ? 1231 : 1237);
		result = prime * result + Arrays.hashCode(states);
		return result;
	}
	/**
	 * This method compares a ComposedState with another object
	 * 
	 * @param obj the other object used by the comparison
	 * @return Return true if and only if obj is a ComposedState and the attributes of the two ComposedState are equals.
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (getClass() != obj.getClass())
			return false;
		ComposedStateImpl other = (ComposedStateImpl) obj;
		if (!super.getName().equals(other.getName()))
			return false;
		if (isFinal != other.isFinal)
			return false;
		if (isInitial != other.isInitial)
			return false;
		if (!Arrays.equals(states, other.states))
			return false;
		return true;
	}

	@Override
	public int compareNumberDifferentStates(ComposedState otherState) {
		State[] otherStatesArray=otherState.getStates();
		int result=0;
		if (otherStatesArray.length!= states.length)
			return -1;
		for (int i=0;i<states.length;i++)
			if(!states[i].getName().equals(otherStatesArray[i].getName()))
				result++;
		
		return result;
	}

}
